/**
 * \file: AudioUtilsHooks.cpp
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * <brief description>.
 * <detailed description>
 * \component: CarPlay
 *
 * \author: J. Harder / ADIT/SW1 / jharder@de.adit-jv.com
 *
 * \copyright (c) 2013-2014 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 * \see <related items>
 *
 * \history
 *
 ***********************************************************************/

// This file is a modified copy of AudioUtilsStub.c
/*
 Copyright (C) 2012-2013 Apple Inc. All Rights Reserved. Not to be used or disclosed without permission from Apple.
 */


#include "Common.h"
#include "AirPlayHeaders.h"

#include "AudioUtils.h"
#include "APSAudioSession.h"

#include <errno.h>
#include <stdlib.h>

#include "CommonServices.h"
#include "DebugServices.h"

#include <dipo_macros.h>
#include "utils/Utils.h"

#include "audio/AudioChannel.h"

#include<inttypes.h>

#include "Session.h"

using namespace adit::carplay;
using namespace std;

#include CF_HEADER
#include LIBDISPATCH_HEADER

typedef struct AudioStreamPrivate * AudioStreamImpRef;
struct CARPLAY_HIDDEN AudioStreamPrivate
{
    CFRuntimeBase base; // CF type info. Must be first.

    Boolean prepared; // True if AudioStreamPrepare has been called (and stop hasn't yet).
    Boolean stop;
    AudioStreamBasicDescription format; // Format of the audio data.
    uint32_t preferredLatencyMics; // Max latency the app can tolerate.
    AudioChannel*       channel;
    bool                hasInput;
    char                audioType[256];
    AirPlayStreamType   streamType;
    double              vocoderSampleRate;
};

#if( AUDIO_STREAM_DLL )
#define _AudioStreamGetImp( STREAM )		( (AudioStreamImpRef) AudioStreamGetContext( (STREAM) ) )
#else
#define _AudioStreamGetImp( STREAM )		(STREAM)
#endif

#if( !AUDIO_STREAM_DLL )
static void
_AudioStreamGetTypeID(void *inContext);
static void
_AudioStreamFinalize(CFTypeRef inCF);
#endif

static APSAudioSessionAudioFormat _audioSessionGetSupportedAudioFormats(AudioStreamType inStreamType, CFStringRef inAudioType);
static APSAudioSessionAudioFormat _audioSessionGetSupportedCompatibililtyAudioFormats(AudioStreamType inStreamType, bool isInput);

static APSAudioSessionAudioFormat _convertSupportedAudioFormats(AudioFormats formats);

#if( !AUDIO_STREAM_DLL )
static dispatch_once_t gAudioStreamInitOnce = 0;
static CFTypeID gAudioStreamTypeID = _kCFRuntimeNotATypeID;
static const CFRuntimeClass kAudioStreamClass =
  { 0, // version
      "AudioStream", // className
      NULL, // init
      NULL, // copy
      _AudioStreamFinalize, // finalize
      NULL, // equal -- NULL means pointer equality.
      NULL, // hash  -- NULL means pointer hash.
      NULL, // copyFormattingDesc
      NULL, // copyDebugDesc
      NULL, // reclaim
      NULL // refcount
    };
#endif
//===========================================================================================================================
//	Logging
//===========================================================================================================================
ulog_define( AudioStream, kLogLevelTrace, kLogFlags_Default, "AudioStream", NULL );
//#define as_dlog( LEVEL, ... )		dlogc( &log_category_from_name( AudioStream ), (LEVEL), __VA_ARGS__ )
//#define as_ulog( LEVEL, ... )		ulog( &log_category_from_name( AudioStream ), (LEVEL), __VA_ARGS__ )

#define kAudioStreamProperty_VocoderSampleRate		CFSTR( "vocoderSampleRate" )

CFTypeID CARPLAY_EXPORTED AudioStreamGetTypeID(void)
{
    dispatch_once_f(&gAudioStreamInitOnce, NULL, _AudioStreamGetTypeID);
    return (gAudioStreamTypeID);
}

static void _AudioStreamGetTypeID(void *inContext)
{
    (void) inContext;

    gAudioStreamTypeID = _CFRuntimeRegisterClass(&kAudioStreamClass);
    check(gAudioStreamTypeID != _kCFRuntimeNotATypeID);
}

OSStatus CARPLAY_EXPORTED AudioStreamCreate(AudioStreamRef* outStream)
{

    auto session = Session::Get(nullptr);
    if (session == nullptr || session->GetConfig() == nullptr)
    {
        LOG_ERROR((dipo, "cannot create audio stream, session is invalid or not initialized"));
        return kUnknownErr;
    }

    // create CF runtime object
    AudioStreamRef me;
    size_t extraLen = sizeof(*me) - sizeof(me->base);
    me = (AudioStreamRef) _CFRuntimeCreateInstance(nullptr, AudioStreamGetTypeID(),
            (CFIndex) extraLen, nullptr);
    dipo_exit_on_null(me);
    memset(((uint8_t *) me) + sizeof(me->base), 0, extraLen);

    // create audio channel
    auto channel = AudioChannel::Add(me);
    channel->SetSession(session);

    // set as context
    me->channel = channel;
    channel->Initialize(*session->GetConfig());

    *outStream = me;
    return kNoErr;
}

static void _AudioStreamFinalize(CFTypeRef inCF)
{
    auto me = (AudioStreamPrivate*)(inCF);
    dipo_return_on_invalid_argument(dipo, me == nullptr);

    AudioChannel::Remove(me);
}

//===========================================================================================================================
//	AudioStreamSetInputCallback
//===========================================================================================================================

void CARPLAY_EXPORTED AudioStreamSetInputCallback(AudioStreamRef inStream,
        AudioStreamInputCallback_f inFunc, void *inContext)
{
    AudioStreamPrivate* me = static_cast<AudioStreamPrivate*>(inStream);
    dipo_return_on_invalid_argument(dipo, inStream == nullptr);
    me->channel->SetInputCallback(inFunc, inContext);
}

//===========================================================================================================================
//	AudioStreamSetOutputCallback
//===========================================================================================================================

void CARPLAY_EXPORTED AudioStreamSetOutputCallback(AudioStreamRef inStream,
        AudioStreamOutputCallback_f inFunc, void *inContext)
{
    AudioStreamPrivate* me = static_cast<AudioStreamPrivate*>(inStream);
    dipo_return_on_invalid_argument(dipo, inStream == nullptr);
    me->channel->SetOutputCallback(inFunc, inContext);
}

//===========================================================================================================================
//	_AudioStreamCopyProperty
//===========================================================================================================================

CFTypeRef _AudioStreamCopyProperty(CFTypeRef inObject, CFObjectFlags inFlags,
        CFStringRef inProperty, CFTypeRef inQualifier, OSStatus * outErr)
{
    AudioStreamImpRef const me = _AudioStreamGetImp((AudioStreamRef) inObject);
    OSStatus err = kNoErr;
    CFTypeRef value = NULL;
    int64_t s64;

    (void) inFlags;
    (void) inQualifier;

    if (0)
    {
    }
    // AudioType
    else if (CFEqual(inProperty, kAudioStreamProperty_AudioType))
    {
        value = CFStringCreateWithCString(nullptr, me->audioType, kCFStringEncodingUTF8);

    }
    // StreamType
    else if (CFEqual(inProperty, kAudioStreamProperty_StreamType))
    {
        value = CFNumberCreateInt64((int64_t)me->streamType);

    }
    // Format
    else if (CFEqual(inProperty, kAudioStreamProperty_Format))
    {
        value = CFDataCreate(NULL, (const uint8_t *) &me->format, sizeof(me->format));

    }
    // Input
    else if (CFEqual(inProperty, kAudioStreamProperty_Input))
    {
        value = me->hasInput ? kCFBooleanTrue : kCFBooleanFalse;
        CFRetain(value);
    }

  // Latency
#if 0
  else if (CFEqual(inProperty, kAudioStreamProperty_Latency))
    {
      s64 = me->channel->GetLatencyMs() * 1000; // convert from ms to micro seconds

      value = CFNumberCreateInt64(s64);
      require_action( value, exit, err = kNoMemoryErr );
    }
#endif
  // PreferredLatency

    else if (CFEqual(inProperty, kAudioStreamProperty_PreferredLatency))
    {
      /* TODO replaced by ADIT
      value = CFNumberCreateInt64(me->preferredLatencyMics);
      */
        s64 = me->channel->GetLatencyMs() * 1000; // convert from ms to micro seconds

        value = CFNumberCreateInt64(s64);

    }

  // ThreadName

    else if (CFEqual(inProperty, kAudioStreamProperty_ThreadName))
    {
      // $$$ TODO: If your implementation uses a helper thread, return its name here.
    }

  // ThreadPriority

    else if (CFEqual(inProperty, kAudioStreamProperty_ThreadPriority))
    {
      // $$$ TODO: If your implementation uses a helper thread, return its priority here.
    }

  // TODO streamType

  // Other

    else
    {
        err = kNotHandledErr;

    }

    if( value == NULL )
    {
  	  LOG_ERROR((dipo, "No Memory Error in while copying the AudioStream Property"));
    }

    if (outErr)
        *outErr = err;
    return (value);
}

//===========================================================================================================================
//	_AudioStreamSetProperty
//===========================================================================================================================

OSStatus _AudioStreamSetProperty(CFTypeRef inObject, CFStringRef inProperty, CFTypeRef inValue)
{
    AudioStreamImpRef const me = _AudioStreamGetImp((AudioStreamRef)inObject);
    OSStatus err = kNoErr;

    // Properties may only be set before AudioStreamPrepare is called.
    if(me->prepared)
    {
    	LOG_ERROR((dipo, "KStateErr Error in AudioStreamSetProperty"));
    }


    if (0)
    {
    }
    // AudioType
    else if (CFEqual(inProperty, kAudioStreamProperty_AudioType))
    {
        // Use the audio type to enable certain types of audio processing.
        // For example, if the audio type is "telephony", echo cancellation should be enabled;
        // if the audio type is "speech recognition", non-linear processing algorithms should be disabled.
        CFGetCString(inValue, me->audioType, sizeof(me->audioType));
    }
    // StreamType
    else if (CFEqual(inProperty, kAudioStreamProperty_StreamType))
    {
        me->streamType = CFGetInt64(inValue, nullptr);
    }
    // Format
    else if (CFEqual(inProperty, kAudioStreamProperty_Format))
    {
        CFGetData(inValue, &me->format, sizeof(me->format), NULL, &err);

    }
    // Input
    else if (CFEqual(inProperty, kAudioStreamProperty_Input))
    {
        me->hasInput = CFGetBoolean(inValue, nullptr);
    }

  // PreferredLatency

  else if (CFEqual(inProperty, kAudioStreamProperty_PreferredLatency))
    {
      me->preferredLatencyMics = (uint32_t) CFGetInt64(inValue, &err);

    }

  // ThreadName

  else if (CFEqual(inProperty, kAudioStreamProperty_ThreadName))
    {
      // $$$ TODO: If your implementation uses a helper thread, set the name of the thread to the string passed in
      // to this property.  See SetThreadName().
    }

  // ThreadPriority

  else if (CFEqual(inProperty, kAudioStreamProperty_ThreadPriority))
    {
      // $$$ TODO: If your implementation uses a helper thread, set the priority of the thread to the string passed in
      // to this property.  See SetCurrentThreadPriority().
    }

  else if (CFEqual(inProperty, kAudioStreamProperty_VocoderSampleRate))
    {
	  me->vocoderSampleRate = CFGetDouble(inValue, &err);
    }

  // Other

  else
    {
      err = kNotHandledErr;
    }

    if ( err != kNoErr )
    {
    	LOG_ERROR((dipo, "Error while setting property to AudioStream"));
    }

  return (err);
}

OSStatus CARPLAY_EXPORTED AudioStreamRampVolume(AudioStreamRef inStream,
        double inFinalVolume, double inDurationSecs, dispatch_queue_t inQueue)
{
#if 0 // TODO to be checked if still valid
    (void)inQueue;
    LOG_INFO((dipo, "AudioStreamRampVolume"));

    dipo_return_value_on_invalid_argument(dipo, inStream == 0, kParamErr);
    auto me = static_cast<AudioStreamPrivate*>(inStream);

    auto session = me->channel->GetSession();
    session->RampVolume(inFinalVolume, inDurationSecs);
#else
    (void)inStream;
    (void)inFinalVolume;
    (void)inDurationSecs;
    (void)inQueue;
#endif

    return kNoErr;
}

#if 0
#pragma mark -
#endif

//===========================================================================================================================
//	AudioStreamPrepare
//===========================================================================================================================

OSStatus CARPLAY_EXPORTED AudioStreamPrepare(AudioStreamRef inStream)
{
    dipo_return_value_on_invalid_argument(dipo, inStream == 0, kParamErr);
    auto me = static_cast<AudioStreamPrivate*>(inStream);

    // This is where the audio processing chain should be set up based on the properties previously set on the
    // AudioStream object:
    //	me->format specifies the sample rate, channel count, and bit-depth.
    //	me->input specifies whether or not the processing chain should be set up to record audio from the accessory's
    //	          microphone(s).
    // Audio output should always be prepared.
    // If the audio processing chain is successfully set up, me->prepared should be set to true.

    AudioFormatStruct format;
    format.SampleRate = me->format.mSampleRate;
    format.Channels = me->format.mChannelsPerFrame;
    format.BitsPerChannel = me->format.mBitsPerChannel;

    // vocoder sample rate in case of telephony wireless cplay.
    if (me->vocoderSampleRate >= 0)
    {
        format.vocoderSampleRate = me->vocoderSampleRate;
    }
    else
    {
        LOG_WARN((dipo, "Received negative value for vocoder sample rate : %f", me->vocoderSampleRate));
        format.vocoderSampleRate = -1.0;
    }

    AudioChannelType channelType = AudioChannelType_Main;
    if (me->streamType == kAirPlayStreamType_AltAudio)
        channelType = AudioChannelType_Alternate;

    if (!me->channel->Prepare(channelType, me->hasInput, format, me->audioType))
        return kUnknownErr;

    me->prepared = true;

    // set audiochannel for flushaudio
    if(me->streamType == kAirPlayStreamType_MainHighAudio)
    {
        auto session = Session::Get(nullptr);
        if (session == nullptr || session->GetConfig() == nullptr)
        {
           LOG_ERROR((dipo, "failed to set AudioChannel for flushAudio, session invalid or not initialized"));
           return kUnknownErr;
        }
        session->SetAudioChannel(me->channel);
    }

    //if (err)
    //    AudioStreamStop(inStream, false);
    return kNoErr;
}

OSStatus CARPLAY_EXPORTED AudioStreamStart(AudioStreamRef inStream)
{
    auto me = static_cast<AudioStreamPrivate*>(inStream);
    dipo_return_value_on_invalid_argument(dipo, inStream == nullptr, kParamErr);
    auto channel = me->channel;

    OSStatus err = kNoErr;

    if (!me->prepared)
    {
        err = AudioStreamPrepare(inStream);
        if ( err != kNoErr )
        {
             LOG_ERROR((dipo, "AudioStreamPrepare Failed In AudioStreamSetProperty"));
        }

    }
    me->stop = false;

    channel->Start();

    // $$$ TODO: This is where the audio processing chain should be started.
    //
    // me->outputCallbackPtr should be invoked periodically to retrieve a continuous stream of samples to be output.
    // When calling me->outputCallbackPtr(), a buffer is provided for the caller to write into.  Equally important
    // is the inSampleTime and inHostTime arguments.  It is important that accurate { inSampleTime, inHostTime } pairs
    // be provided to the caller.  inSampleTime should be a (reasonably) current running total of the number of samples
    // that have hit the speaker since AudioStreamStart() was called.  inHostTime is the system time, in units of ticks,
    // corresponding to inSampleTime (see TickUtils.h).  This information will be returned to the controller and is
    // a key piece in allowing the controller to derive the relationship between the controller's system clock and the
    // accessory's audio (DAC) clock for A/V sync.
    //
    // If input has been requested (me->input == true), then me->inputCallbackPtr should also be invoked periodically
    // to provide a continuous stream of samples from the accessory's microphone (possibly with some processing, depending
    // on the audio type, see kAudioStreamProperty_AudioType).  If no audio samples are available for whatever reason,
    // the me->inputCallbackPtr should be called with a buffer of zeroes.


    if (err)
        AudioStreamStop(inStream, false);
    return (err);
}

//===========================================================================================================================
//	AudioStreamStop
//===========================================================================================================================

void CARPLAY_EXPORTED AudioStreamStop(AudioStreamRef inStream, Boolean inDrain)
{
    auto me = static_cast<AudioStreamPrivate*>(inStream);
    dipo_return_on_invalid_argument(dipo, inStream == nullptr);
    auto channel = me->channel;

    // $$$ TODO: This is where the audio processing chain should be stopped, and the audio processing chain torn down.
    // When AudioStreamStop() returns, the object should return to the state similar to before AudioStreamPrepare()
    // was called, so this function is responsible for undoing any resource allocation performed in AudioStreamPrepare().
    (void) inDrain;

    me->stop = true;
    me->prepared = false;

    channel->Stop();
}

void CARPLAY_EXPORTED APSAudioSessionSetEventHandler(APSAudioSessionEventHandler_f inHandler,
    void* inContext)
{
    (void)inHandler;
    (void)inContext;

    // This implementation should remain empty.
}

void CARPLAY_EXPORTED APSAudioSessionEnsureSetup(Boolean hasInput,
        uint32_t inPreferredSystemSampleRate, uint32_t inPreferredSystemBufferSizeSamples)
{
    (void)hasInput;
    (void)inPreferredSystemSampleRate;
    (void)inPreferredSystemBufferSizeSamples;

    // This implementation should remain empty.
}

void CARPLAY_EXPORTED APSAudioSessionEnsureTornDown(void)
{
    // This implementation should remain empty.
}

APSAudioSessionAudioFormat CARPLAY_EXPORTED APSAudioSessionGetSupportedFormats(AudioStreamType inStreamType, CFStringRef inAudioType)
{
	return _audioSessionGetSupportedAudioFormats(inStreamType, inAudioType);
}

APSAudioSessionAudioFormat CARPLAY_EXPORTED APSAudioSessionGetCompatibilityInputFormats(AudioStreamType inStreamType)
{
	return _audioSessionGetSupportedCompatibililtyAudioFormats(inStreamType, true);
}
APSAudioSessionAudioFormat CARPLAY_EXPORTED APSAudioSessionGetCompatibilityOutputFormats(AudioStreamType inStreamType)
{
	return _audioSessionGetSupportedCompatibililtyAudioFormats(inStreamType, false);
}

CFArrayRef CARPLAY_EXPORTED APSAudioSessionCopyLatencies( OSStatus *outErr )
{
    OSStatus err = kNoErr;

    // Obtain audio latencies for all audio formats and audio types supported by the underlying hardware.
    // Audio latencies are reported as an ordered array of dictionaries (from least restrictive to the most restrictive).
    // Each dictionary contains the following keys:
    //      [kAPSAudioSessionKey_Type] - if not specified, then latencies are good for all stream types
    //      [kAPSAudioSessionKey_AudioType] - if not specified, then latencies are good for all audio types
    //      [kAPSAudioSessionKey_SampleRate] - if not specified, then latencies are good for all sample rates
    //      [kAPSAudioSessionKey_SampleSize] - if not specified, then latencies are good for all sample sizes
    //      [kAPSAudioSessionKey_Channels] - if not specified, then latencies are good for all channel counts
    //      [kAPSAudioSessionKey_CompressionType] - if not specified, then latencies are good for all compression types
    //      kAPSAudioSessionKey_InputLatencyMicros
    //      kAPSAudioSessionKey_OutputLatencyMicros

    auto session = Session::Get(nullptr);
    if (session == nullptr)
    {
        LOG_ERROR((dipo, "unable to get audio latencies"));
        *outErr = kUnknownErr;
        return nullptr;
    }

    auto config = session->GetConfig();
    auto list = config->GetItems("audio-latency-micros");

    CFMutableArrayRef array = CFArrayCreateMutable(nullptr, 0, &kCFTypeArrayCallBacks);
    dipo_exit_on_null(array);

    for (auto line : list)
    {
        const CFFromStdPair keyTranslation[] = {
                { "audiotype", kAPSAudioSessionKey_AudioType, CFFromStdType_String },
                { "input", kAPSAudioSessionKey_InputLatencyMicros, CFFromStdType_Number },
                { "output", kAPSAudioSessionKey_OutputLatencyMicros, CFFromStdType_Number },
                { "type", kAPSAudioSessionKey_Type, CFFromStdType_Number },
                { "samplerate", kAPSAudioSessionKey_SampleRate, CFFromStdType_Number },
                { "samplesize", kAPSAudioSessionKey_SampleSize, CFFromStdType_Number },
                { "channels", kAPSAudioSessionKey_Channels, CFFromStdType_Number },
                { "compressiontype", kAPSAudioSessionKey_CompressionType, CFFromStdType_Number }
        };

        map<string, string> latencyMap = ParseMap(line, true, "audio-latency-micros");

        // Change Audio stream type to Apple specified values
        map<string,string>::iterator iter = latencyMap.find("type");
        if(iter != latencyMap.end())
        {
            switch(std::stoi(iter->second))
            {
            case AudioChannelType_Main:
                iter->second = "100";
                break;
            case AudioChannelType_Alternate:
                iter->second = "101";
                break;
            case AudioChannelType_Main_High:
                iter->second = "102";
                break;
            default:
                LOG_ERROR((dipo, "audio type %s not supported", iter->second.c_str()));
                err = kUnknownErr;
                break;
            }

            if(err == kNoErr)
            {
                auto latency = CFDictionaryFromStdMap(latencyMap, keyTranslation,
                        sizeof(keyTranslation) / sizeof(keyTranslation[0]), "audio-latency-micros");
                if(latency != nullptr)
                {
                    CFArrayAppendValue(array, latency);
                }
                if (outErr) *outErr = err;
            }
        }
    }

    return array;
}

// ========== private functions ==========

APSAudioSessionAudioFormat _audioSessionGetSupportedAudioFormats(AudioStreamType inStreamType, CFStringRef inAudioType)

{
    AudioFormats formats = 0;
    auto session = Session::Get(nullptr);
    if (session != nullptr)
    {
        /* $TODO: see j.harder's comment in CR-SOPL-10
         * usage of map will reduce the code size*/
        switch( inStreamType )
        {
            case kAudioStreamType_MainAudio:
                if( CFEqual( inAudioType, CFSTR( kAirPlayAudioType_Default ) ) )
                {
                    formats = session->GetAudioFormats(AudioChannelType_Main,AudioStreamType_Default);
                    LOGD_DEBUG((dipo, "MainAudio Default type formats supported by HW:0x%" PRIx64,
                                    (uint64_t)formats));

                }
                else if( CFEqual( inAudioType, CFSTR( kAirPlayAudioType_Media ) ) )
                {
                    formats = session->GetAudioFormats(AudioChannelType_Main,AudioStreamType_Media);
                    LOGD_DEBUG((dipo, "MainAudio Media type formats supported by HW:0x%" PRIx64,
                                      (uint64_t)formats));
                }
                else if( CFEqual( inAudioType, CFSTR( kAirPlayAudioType_Telephony ) ) )
                {
                    formats = session->GetAudioFormats(AudioChannelType_Main,AudioStreamType_Telephony);
                    LOGD_DEBUG((dipo, "MainAudio Telephony type formats supported by HW:0x%" PRIx64,
                                      (uint64_t)formats));

                }
                else if( CFEqual( inAudioType, CFSTR( kAirPlayAudioType_SpeechRecognition ) ) )
                {
                    formats = session->GetAudioFormats(AudioChannelType_Main,AudioStreamType_SpeechRecognition);
                    LOGD_DEBUG((dipo, "MainAudio SpeechRecognition type formats supported by HW:0x%" PRIx64,
                                      (uint64_t)formats));
                }
                else if( CFEqual( inAudioType, CFSTR( kAirPlayAudioType_Alert ) ) )
                {
                    formats = session->GetAudioFormats(AudioChannelType_Main,AudioStreamType_Alert);
                    LOGD_DEBUG((dipo, "MainAudio Alert type formats supported by HW:0x%" PRIx64,
                                      (uint64_t)formats));
                }
                else
                {
                	char audioType[1024];
                	CFGetCString(inAudioType, audioType, sizeof(audioType));
                	LOG_ERROR((dipo, "Unsupported Audio type \"%s\" received for main audio", audioType));
                }
                break;

            case kAudioStreamType_MainHighAudio:
                if( CFEqual( inAudioType, CFSTR( kAirPlayAudioType_Media ) ) )
                {
                    formats = session->GetAudioFormats(AudioChannelType_Main_High,AudioStreamType_Media);
                    LOGD_DEBUG((dipo, "Main High Audio Media type formats supported by HW:0x%" PRIx64,
                                      (uint64_t)formats));
                }
                break;

            case kAudioStreamType_AltAudio:
                if( CFEqual( inAudioType, CFSTR( kAirPlayAudioType_Default ) ) )
                {
                    formats = session->GetAudioFormats(AudioChannelType_Alternate,AudioStreamType_Default);
                    LOGD_DEBUG((dipo, "Alternate Audio Default type formats supported by HW:0x%" PRIx64,
                                      (uint64_t)formats));
                }
                break;

            default:
                formats = 0;
                break;
        }
    }
    else
    {
        LOG_ERROR((dipo, "unable to get supported audio formats, no session active"));
    }

    return _convertSupportedAudioFormats(formats);
}

APSAudioSessionAudioFormat _audioSessionGetSupportedCompatibililtyAudioFormats(AudioStreamType inStreamType, bool isInput)
{
    AudioFormats formats = 0;
    auto session = Session::Get(nullptr);
    if (session != nullptr)
    {
        switch( inStreamType )
        {
            case kAudioStreamType_MainAudio:
                if(isInput)
                {
                    formats = session->GetCompatibilityAudioFormats(true, AudioChannelType_Main, AudioStreamType_Compatibility);
                    LOGD_DEBUG((dipo, "MainAudio Compatibility type Output formats supported by HW: 0x%" PRIx64,
                                (uint64_t)formats));
                }
                else
                {
                    formats = session->GetCompatibilityAudioFormats(false, AudioChannelType_Main, AudioStreamType_Compatibility);
                    LOGD_DEBUG((dipo, "MainAudio Compatibility type Input formats supported by HW: 0x%" PRIx64,
                                (uint64_t)formats));
                }
                break;
            case kAudioStreamType_MainHighAudio:
                LOG_ERROR((dipo, "compability type not supported for mainHigh Audio"));
                break;
            case kAudioStreamType_AltAudio:
                if(!isInput)
                {
                    formats = session->GetCompatibilityAudioFormats(false, AudioChannelType_Alternate, AudioStreamType_Compatibility);
                    LOGD_DEBUG((dipo, "Alternate Audio Compatibility type formats supported by HW:0x%" PRIx64,
                                      (uint64_t)formats));
                }
                break;
            default:
                formats = 0;
                break;
        }
    }
    else
    {
        LOG_ERROR((dipo, "unable to get supported audio formats, no session active"));
    }

    return _convertSupportedAudioFormats(formats);
}

APSAudioSessionAudioFormat _convertSupportedAudioFormats(AudioFormats formats)
{
    if (sizeof(APSAudioSessionAudioFormat) != sizeof(AudioFormats))
    {
        LOG_ERROR((dipo, "sizeof(APSAudioSessionAudioFormat) != sizeof(AudioFormats)"));
        return 0;
    }
    return (APSAudioSessionAudioFormat)(uint64_t)(formats << 2);
}
